; Internal representation of assembler programs and display routines
; ===========================================================================

; ---------------------------------------------------------------------------
; The object types that make up the internal assembly language representation.
; ---------------------------------------------------------------------------

; Object type command sequence (seq). This represents a program.
(define seq (constructor 'seq))
(define seq? (detector 'seq))
(define seq-nargs object-size)
(define seq-args cdr)

; optional program
(define (opt test prog) (if test prog (seq)))

; ........................................
; The types for the different commands.
; ........................................

; Object type instruction (inst).
; Its first member is the operation, a string.
; All remaining members are arguments to the operation.
(define inst (constructor 'inst))
(define inst? (detector 'inst))
(define inst-op (getter 0))
(define inst-nargs (lambda (i) (- (object-size i) 1)))
(define inst-arg (lambda (n i) ((getter (+ n 1)) i)))
(define (inst-args o) (cdr (object-members o)))

; Object type label.
; It has one member, a string, the name of the label.
(define label (constructor 'label))
(define label? (detector 'label))
(define label-symbol (getter 0))
(define (label-string l) (symbol->string (label-symbol l)))

; Object type definition (def)
; It has two members, a name object and its value, an operand (not any type)
(define def (constructor 'def))
(define def? (detector 'def))
(define def-name (getter 0))
(define def-value (getter 1))

; Object type data
; It has two members, a size in bytes and a numeric value
(define data (constructor 'data))
(define data? (detector 'data))
(define data-size (getter 0))
(define data-value (getter 1))

; Object type text
; It has one member, the string (no newline) to be displayed.
(define text (constructor 'text))
(define text? (detector 'text))
(define text-string (getter 0))

; Object type comment.
; It has one member, the comment string, which must be a single-line string.
(define comment (constructor 'comment))
(define comment? (detector 'comment))
(define comment-string (getter 0))

; Object type entry.
; It has one member, a label.
(define entry (constructor 'entry))
(define entry? (detector 'entry))
(define entry-label (getter 0))

; ........................................
; The types for arguments to instructions. (operands)
; ........................................

; Object type operand size (op-size).
; It has 2 members, the integer size indicator (in bytes) and the operand.
(define op-size (constructor 'op-size))
(define op-size? (detector 'op-size))
(define op-size-size (getter 0))
(define op-size-op (getter 1))

; Object type register.
; It has one member, a symbol, the name of the register.
(define reg (constructor 'reg))
(define reg? (detector 'reg))
(define reg-symbol (getter 0))
(define (reg-string r)
  (list->string (map char-upcase
		     (string->list (symbol->string (reg-symbol r))))))

; Object type index.
; It has any positive number of members. Which are non-label operands.
; Its semantics is that all arguments are summed and the result is
; dereferenced. This allows many types of indexing.
(define index (constructor 'index))
(define index? (detector 'index))
(define index-nargs object-size)
(define index-args object-members)
; A simple dereference is when there is nothing to sum (just one arg).
(define (deref? arg) (and (index? arg) 
			  (= (object-size arg) 1)))

; Object type name.
; It has any number of members, symbols, which form a big name.
(define name (constructor 'name))
(define name? (detector 'name))
(define (name-string n) 
  (let ((strings (map symbol->string (object-members n))))
    (apply string-append 
	   (cons (car strings)
		 (map (lambda (x) (string-append "_" x)) (cdr strings))))))

; Object type binary operator (binop)
; It has 3 members : the operator string and the two arguments
(define binop (constructor 'binop))
(define binop? (detector 'binop))  
(define binop-opstring (getter 0))
(define binop-arg1 (getter 1))
(define binop-arg2 (getter 2))

; Object type label has alread been defined.
; It is mentionned here because a label may be an operand.

; Numbers are also a valid type of operand.

; ........................................
; Another type that can appear as an index arg.
; ........................................

; Object type scale.
; It has two members. The first is a register and the other is a number.
(define scale (constructor 'scale))
(define scale? (detector 'scale))
(define scale-reg (getter 0))
(define scale-scale (getter 1))

; ---------------------------------------------------------------------------
; Display routines
; ---------------------------------------------------------------------------

; Newline string.
(define nl "\n")

; Tab string.
(define tab "\t")

; Take a program, a language identifier and optionally a port and
; writes the program in the given language on the specified port.
; The port returned by (current-output-port) is the default port.
(define (display-asm program language . rest)
  (let* ((port (if (pair? rest) (car rest) (current-output-port)))
	 (p (lambda args (for-each (lambda (x) (display x port)) args)))
	 (line-comment-prefix
	  (case language
	    ((as86) ";")
	    (else (Error "Unsupported language :" language))))
	 (display-comment (lambda (string)
			    (p line-comment-prefix " " string nl))))
    (display-comment
     "Assembler file generated by the Scheme Tunes LLL generator")
    (p nl)
    (letrec ((display-index-arg
	      (lambda (arg)
		(case (type arg)
		  ((number) (p arg))
		  ((reg) (p (reg-string arg)))
		  ((scale) (p (reg-string (scale-reg arg))
			      "*" (scale-scale arg)))
		  (else 
		   (error "Tried to write an unsupported indexing argument"
			  arg)))))
	     (display-arg
	      (lambda (arg)
		(case (type arg)
		  ((number) (p arg))
		  ((op-size) (p (case (op-size-size arg)
				  ((1) "byte ")
				  ((2) "word ")
				  ((4) "dword ")
				  (else 
				   (error "Wrong size :" (op-size-size arg)))))
			     (display-arg (op-size-op arg)))
		  ((index) (p "[")
			   (display-index-arg (car (index-args arg)))
			   (for-each (lambda (x) 
				       (if (not (and (number? x)
						     (negative? x)))
					   (p "+"))
				       (display-index-arg x))
				     (cdr (index-args arg)))
			   (p "]"))
		  ((binop) (p "(") 
			   (display-arg (binop-arg1 arg))
			   (p " " (binop-opstring arg) " ")
			   (display-arg (binop-arg2 arg))
			   (p ")"))
		  ((reg) (p (reg-string arg)))
		  ((name) (p (name-string arg)))
		  ((label) (p (label-string arg))))))
	     (display-args
	      (lambda (args)
		(if (pair? args)
		    (begin (display-arg (car args))
			   (if (pair? (cdr args))
			       (begin (p ", ")
				      (display-args (cdr args))))))))
	     (display-command 
	      (lambda (c)
		(case (type c)
		  ((seq) (for-each display-command (seq-args c)))
		  ((inst) (p tab (inst-op c) tab)
			  (display-args (inst-args c))
			  (p nl))
		  ((data) (p tab
			     (case (data-size c)
			       ((1) ".byte ")
			       ((2) ".word ")
			       ((4) ".dword ")
			       (else (error "Wrong size :" (data-size c))))
			     tab (data-value c) nl))
		  ((label) (p (label-string c) ":" nl))
		  ((entry) (p "entry" tab (label-string (entry-label c)) nl))
		  ((def) (p (name-string (def-name c)) " = ")
			 (display-arg (def-value c))
			 (p nl))
		  ((text) (p (text-string c) nl))
		  ((struc) (display-command (compile-struc c)))
		  ((comment) (display-comment (comment-string c)))
		  (else (error "Unrecognised command :" c))))))
      (display-command program))))

; ---------------------------------------------------------------------------
; Definition of registers to make the assembler writen in Scheme look better.
; ---------------------------------------------------------------------------

(define ah '(reg ah))
(define bh '(reg bh))
(define ch '(reg ch))
(define dh '(reg dh))
(define al '(reg al))
(define bl '(reg bl))
(define cl '(reg cl))
(define dl '(reg dl))
(define ax '(reg ax))
(define bx '(reg bx))
(define cx '(reg cx))
(define dx '(reg dx))
(define eax '(reg eax))
(define ebx '(reg ebx))
(define ecx '(reg ecx))
(define edx '(reg edx))
(define bp '(reg bp))
(define sp '(reg sp))
(define si '(reg si))
(define bi '(reg bi))
(define ebp '(reg ebp))
(define esp '(reg esp))
(define esi '(reg esi))
(define ebi '(reg ebi))
(define cs '(reg cs))
(define ds '(reg ds))
(define es '(reg es))
(define fs '(reg fs))
(define gs '(reg gs))
(define ss '(reg ss))
(define eip '(reg eip))
(define eflags '(reg eflags))
(define cr0 '(reg cr0))
(define cr1 '(reg cr1))
(define cr2 '(reg cr2))
(define cr3 '(reg cr3))
(define dr0 '(reg dr0))
(define dr1 '(reg dr1))
(define dr2 '(reg dr2))
(define dr3 '(reg dr3))
(define dr4 '(reg dr4))
(define dr5 '(reg dr5))
(define dr6 '(reg dr6))
(define dr7 '(reg dr7))
(define tr3 '(reg tr3))
(define tr4 '(reg tr4))
(define tr5 '(reg tr5))
(define tr6 '(reg tr6))
(define tr7 '(reg tr7))

; ---------------------------------------------------------------------------
; Functions to link 32 bits registers to their 8 and 16 bits versions
; ---------------------------------------------------------------------------

(define (rconverter n)
  (lambda (r)
    (let ((table '(("EAX" "AX" "AH" "AL")
		   ("EBX" "BX" "BH" "BL")
		   ("ECX" "CX" "CH" "CL")
		   ("EDX" "DX" "DH" "DL")
		   ("EBP" "BP"  ""   "")
		   ("ESP" "SP"  ""   "")
		   ("ESI" "SI"  ""   "")
		   ("EBI" "BI"  ""   ""))))
      (letrec ((conv2 (lambda (t)
			(if (null? t)
			    (error "Can't convert register" r)
			    (if (member (reg-string r) (car t))
				(let ((res (list-ref (car t) n)))
				  (if (equal? res "")
				      (error "Can't convert register" r)
				      res))
				(conv2 (cdr t)))))))
	(reg (string->symbol (conv2 table)))))))

(define r32 (rconverter 0))
(define r16 (rconverter 1))
(define r8h (rconverter 2))
(define r8l (rconverter 3))

; ---------------------------------------------------------------------------
; Label generator
; ---------------------------------------------------------------------------

(define (make-counter)
  (let ((n -1))
    (lambda ()
      (set! n (+ n 1))
      n)))

(define gen-label
  (let ((counter (make-counter)))
    (lambda ()
      (label (string->symbol
	      (string-append "L" (number->string (counter))))))))

