; Control structures functions that generate assembler programs
; ===========================================================================

; ---------------------------------------------------------------------------
; Conditionals
; ---------------------------------------------------------------------------

; Translate from a "user-friendly comparison" to the string that goes
; in the mnemonic instruction.
(define (test-string test)
  (let ((table '((> . "a") (a . "a") (nbe . "a")
		 (< . "b") (b . "b") (nae . "b")
		 (>= . "ae") (=> . "ae") (ae . "ae") (nb . "ae")
		 (<= . "be") (=< . "be") (be . "be") (na . "be")
		 (= . "e") (== . "e") (s= . "e") 
		 (s== . "e") (e . "e") (z . "e")
		 (<> . "ne") (!= . "ne") (~= . "ne") (ne . "ne") (nz . "ne")
		 (s<> . "ne") (s!= . "ne") (s~= . "ne")
		 (s> . "g") (g . "g") (nle . "g")
		 (s< . "l") (l . "l") (nge . "l")
		 (s>= . "ge") (s=> . "ge") (ge . "ge") (nl . "ge")
		 (s<= . "le") (s=< . "le") (le . "le") (ng . "le"))))
    (cdr (assoc test table))))

; Negate a test condition
(define (negate-test test)
  (let ((table '(((> a nbe) <=)
		 ((< b nae) >=)
		 ((>= => ae nb) <)
		 ((<= =< be na) >)
		 ((= == s= s== e z) <>)
		 ((<> != ~= ne nz s<> s!= s~=) =)
		 ((s> g nle) s<=)
		 ((s< l nge) s>=)
		 ((s>= s=> ge nl) s<)
		 ((s<= s=< le ng) s>))))
    (letrec ((lookup (lambda (x t)
		       (if (null? t)
			   #f
			   (if (member x (caar t))
			       (cadar r)
			       (lookup x (cdr t)))))))
      (lookup test table))))

; This function should not be used directly, use if-jump or if-set.
(define (if-jump-or-set op1 test op2 dest action)
  (seq (inst 'cmp op1 op2)
       (inst (string->symbol
	      (string-append action (test-string test))) dest)))

; Jumps to the specified destination if the condition is true.
(define (if-jump op1 test op2 dest)
  (if-jump-or-set op1 test op2 dest "j"))

; Executes prog-part iff the condition is false.
(define (unless op1 test op2 prog-part)
  (let ((l (gen-label)))
    (seq (if-jump op1 test op2 l)
	 prog-part
	 l)))

; Executes then-part iff the condition is true.
(define (if-then op1 test op2 then-part)
  (unless op1 (negate-test test) op2 then-part))

; Executes then-part iff the condition is true and else-part otherwise.
(define (if-then-else op1 test op2 then-part else-part)
  (let ((begin-else (gen-label))
	(end-else (gen-label)))
    (seq (if-jump op1 (negate-test test) op2 begin-else)
	 then-part
	 (inst 'jmp end-else)
	 begin-else
	 else-part
	 end-else)))

; ---------------------------------------------------------------------------
; Loops
; ---------------------------------------------------------------------------

; Execute prog-part at least once and then while the condition is true.
; l1 and l2 are the labels before and after the while.
; They can be used to implement C's continue and break respectively.
(define (while1-labels op1 test op2 prog-part l1 l2)
  (seq l1
       prog-part
       (if-jump op1 test op2 l1)
       l2))

; The while1 to use if you don't want continue or break.
(define (while1 op1 test op2 prog-part)
  (while1-labels op1 test op2 prog-part (gen-label) (gen-label)))


; Execute prog-part at while the condition is true.
; l1 and l2 are the labels before and after the while.
; They can be used to implement C's continue and break respectively.
(define (while-labels op1 test op2 prog-part l1 l2)
  (seq l1
       (if-jump op1 (negate-test test) op2 l2)
       prog-part
       (inst 'jmp l1)
       l2))

; The while to use if you don't want continue or break.
(define (while op1 test op2 prog-part)
  (while-labels op1 test op2 prog-part (gen-label) (gen-label)))


; ---------------------------------------------------------------------------
; Misc.
; ---------------------------------------------------------------------------

; Set to 1 the specified location if the condition is true (0 otherwise).
(define (if-set op1 test op2 loc)
  (if-jump-or-set op1 test op2 loc "set"))

; Call but build the label object before if a symbol was passed
(define (call sym)
  (if (symbol? sym)
      (inst 'call (label sym))
      (inst 'call sym)))