; Data structures functions that generate assembler programs
; ===========================================================================

; Object type data element (elem)
; It has three members, the first is a symbol which names the element
; the second is the size of the units
(define elem (constructor 'elem '(#f 4 1)))
(define elem? (detector 'elem))
(define elem-name (getter 0))
(define elem-unit-size (getter 1))
(define elem-length (getter 2))
(define (elem-size e) (* (elem-unit-size e) (elem-length e)))

; Object type data structure (struc)
; Its first member is a symbol, the name of the structure and the others
; are the components of the structure.
(define struc (constructor 'struc))
(define struc? (detector 'struc))
(define struc-name (getter 0))
(define (struc-components s) (cdr (object-members s)))

; Return a list of constant definitions for the indices
; of the members of the structure.
(define (compile-struc s) (apply seq (car (compile-struc-pair s '() 0))))
(define (compile-struc-pair s prefix start-address) 
  (case (type s)
    ((struc) 
     (if (null? (struc-components s))
	 (cons '() start-address)
	 (let* ((head-pair (compile-struc-pair 
			    (car (struc-components s)) 
			    (append prefix (list (struc-name s)))
			    start-address))
		(head-compiled (car head-pair))
		(head-end-address (cdr head-pair))
		(tail-pair (compile-struc-pair
			    (apply struc (cons (struc-name s)
					       (cdr (struc-components s))))
			    prefix
			    head-end-address))
		(tail-compiled (car tail-pair))
		(tail-end (cdr tail-pair)))
	   (cons (append head-compiled tail-compiled) tail-end))))
    ((elem)
     (let ((address (+ start-address 
		       (modulo (- start-address) (elem-unit-size s)))))
       (cons (list (def (apply name (append prefix (list (elem-name s))))
			address))
	     (+ address (elem-size s)))))
    (else (error "This cannot be part of a structure :" s))))

; Generate a list of similar elements
(define (list-elems size . rest)
  (map (lambda (sym) (elem sym size)) rest))

; Alternative to specifying the size with the number of bytes
(define byte 1)
(define word 2)
(define long 4)
(define dword 4)
