; i386 specific definitions
; ===========================================================================

; ---------------------------------------------------------------------------
; Useful Constants
; ---------------------------------------------------------------------------

(define FLAG_CF		#x00000001) ;	Carry
(define FLAG_SET	#x00000002) ;	????
(define FLAG_PF		#x00000004) ;	Parity
(define FLAG_AF		#x00000010) ;	Auxiliary Carry for BCD
(define FLAG_ZF		#x00000040) ;	Zero
(define FLAG_SF		#x00000080) ;	Sign
(define FLAG_TF		#x00000100) ;	Trap
(define FLAG_IF		#x00000200) ;	Interrupt 0:Disable 1:Enable
(define FLAG_DF		#x00000400) ;	Direction 0:Upward  1:Downward
(define FLAG_OF		#x00000800) ;	Overflow
(define FLAG_IOPL	#x00003000) ;	IO Priviledge requirement mask
(define FLAG_IOPL1	#x00001000) ;	IO Priviledge requirement 1
(define FLAG_IOPL2	#x00002000) ;	IO Priviledge requirement 2
(define FLAG_IOPL3	#x00003000) ;	IO Priviledge requirement 3
(define FLAG_NT		#x00004000) ;	Nested Task
(define FLAG_RF		#x00010000) ;	Resume debug
(define FLAG_VM		#x00020000) ;	V86 Mode
(define FLAG_AC		#x00040000) ;	Alignment Check
(define FLAG_ID		#x00200000) ;	CPUID available

(define CR0_PE	#x00000001) ;	Protected mode Enable
(define CR0_MP	#x00000002) ;	Math copro Present
(define CR0_EM	#x00000004) ;	Emulate Math copro
(define CR0_TS	#x00000008) ;	Task Switched
(define CR0_ET	#x00000010) ;	Extension Type
(define CR0_NE	#x00000020) ;	Numeric Error
(define CR0_WP	#x00010000) ;	Write Protect
(define CR0_AM	#x00020000) ;	Alignment Mask
(define CR0_NW	#x20000000) ;	No Write-back
(define CR0_CD	#x40000000) ;	Cache Disable
(define CR0_PG	#x80000000) ;	Pageing

(define CR3_PWT	#x00000004) ;	Page Write Transparency
(define CR3_PCD	#x00000010) ;	Page Cache Disable


; .........................
; ASCII codes
; .........................
(define LF	#x0A)
(define CR	#x0D)
(define Escape	#x1B)

; .........................
; Keyboard scancodes
; .........................
(define K_Home	#x47)
(define K_Up	#x48)
(define K_PgUp	#x49)
(define K_Lft	#x4B)
(define K_Rgt	#x4D)
(define K_EndC	#x4F)
(define K_Dn	#x50)
(define K_PgDn	#x51)
(define K_InsC	#x52)
(define K_Del	#x53)


; .........................
; Architecture-dependent information
; .........................

(define BYTE-ORDER 1234) ;(little endian)
(define PAGESIZE 4096)
(define PAGEBITS 12)

; ---------------------------------------------------------------------------
; Assembly control information
; ---------------------------------------------------------------------------

; Are we generating code for 32 bits mode? Can be changed with set!
(define mode32 #f)

; Are we generating code for protected mode? Can be changed with set!
(define p-mode #f)

; Choose the appropriate version of the register according to the current mode.
(define (r?? r) ((if mode32 r32 r16) r))

; ---------------------------------------------------------------------------
; Instruction set enhancement macros
; ---------------------------------------------------------------------------
					   
; Determines if this is a register on which most operations can be done
(define (general-reg? r) 
  (and (reg? r) (member (reg-symbol r)
			'(al ah ax eax bh bl bx ebx
			  ch cl cx ecx dh dl dx edx
			  sp bp si di esp ebp esi edi))))

(define (valid-addressing? op) (******))
(define (valid-operand? op) (or (general-reg? op) (valid-addressing? op)))
					   
; Clear all listed operands
(define (clr . ops) 
  (apply seq (map (lambda (op)
		    (if (general-reg? op)
			(inst 'xor op op)
			(inst 'mov (assert valid-addressing? op) 0)))
		  ops)))

; Set all bits of listed operands
(define (setf . ops) 
  (apply seq (map (lambda (op)
		    (if (general-reg? op)
			(inst 'or op op)
			(inst 'mov (assert valid-addressing? op) -1)))
		  ops)))
  
; Changes byte order of a 32 bit general register
(define (bswaperx r)
  (assert general-reg? r)
  (seq (inst 'xchg (r8l r) (r8h r))
       (inst 'rol  (r32 r) 16) 
       (inst 'xchg (r8l r) (r8h r))))
       
(define (flush-instr)
  (let ((l (gen-label)))
    (seq (inst 'jmp l)
	 l)))

(define (die)
  (let ((l (gen-label)))
    (seq l
	 (inst 'hlt)
	 (inst 'jmp l))))

(define (pushfX) (inst (if mode32 'pushfd 'pushf)))
(define (popfX) (inst (if mode32 'popfd 'popf)))
(define (pushaX) (inst (if mode32 'pushad 'pusha)))
(define (popaX) (inst (if mode32 'popad 'popa)))
; * BEWARE !!!
; The instruction following a popa/popad MUST NOT use the base+index register
; addressing mode, or else most 386 cpus will trash eax, or even crash if eax
; is used in the addressing. That's an intel bug. Avoid it.

; Push in reverse order the list of operands
; Apart from operands, symbols a ad ax f fd fx will call push[a|f][d|X]?
(define (pushm . ops) 
  (apply seq (map (lambda (op)
		    (if (equal? op (r?? ip))
			(let ((l (gen-label)))
			  (seq (call l)
			       l))
			(if (symbol? op)
			    (case op
			      ((a) (inst 'pusha))
			      ((ad) (inst 'pushad))
			      ((ax) (pushaX))
			      ((f) (inst 'pushf))
			      ((fd) (inst 'pushfd))
			      ((fx) (pushfX)))
			    (begin (assert valid-operand? op)
				   (inst 'push op)))))
		  (reverse ops))))

; Pop in given order the list of operands (so the same operand list that
; whas used to pushm should be use to popm to undo the pushm)
(define (popm . ops) 
  (apply seq (map (lambda (op)
		    (if (equal? op (r?? ip))
			(inst 'ret)
			(if (symbol? op)
			    (case op
			      ((a) (inst 'popa))
			      ((ad) (inst 'popad))
			      ((ax) (popaX))
			      ((f) (inst 'popf))
			      ((fd) (inst 'popfd))
			      ((fx) (popfX)))
			    (begin (assert valid-operand? op)
				   (inst 'pop op)))))
		  ops)))

(define (pushing operands . program)
  (seq (pushm operands)
       (apply seq program)
       (popm operands)))

; Move if r2 to t1 if they are different
(define (mov r1 r2)
  (if (equal? r1 r2)
      (seq)
      (inst 'mov r1 r2)))

; Move r2 to r1 by pushing/popping (unless r1=r2)
(define (ppmov r1 r2) 
  (if (equal r1 r2)
      (seq)
      (seq (pushm r2) (popm r1))))

; Move r2 to r1 through temporary reg/mem r3
(define (ttmov r1 r2 r3)
  (assert reg? r1)
  (assert reg? r2)
  (assert valid-operand? r3)
  (seq (mov r3 r2)
       (mov r1 r3)))

(define (movzil r op)
  (seq (clr (r32 r))
       (inst 'mov (r8l r) op)))

(define (movm . args)
  (if (null? args)
      (seq)
      (apply seq (map (lambda (x) 
			(assert valid-operand? (car x))
			(inst 'mov (car x) (cadr x)))
		      args))))
      

(define (SegOfsToLinear r1 r2 op1 op2. rest)
  (let ((adjust (if (null? rest) #f (car rest)))
	(update (if (or (null? rest) (null? (cdr rest))) #f (cadr rest))))
    (seq (inst 'movzx r1 (op-size word op2))
	 (inst 'movzx (r32 r2) (op-size word op1))
	 (if adjust (AddIm r2 adjust) (seq))
	 (inst 'shl r1 4)
	 (inst 'add r1 (r32 r2))
	 (if update (inst 'mov op1 r2) (seq)))))


; ---------------------------------------------------------------------------
; Structured programming 
; ---------------------------------------------------------------------------

; Most of this has been defined in "control_structures.scm"

; ---------------------------------------------------------------------------
; String definitions
; ---------------------------------------------------------------------------

; This currently represents strings as a bunch of bytes not as assembler
; strings. Replace #t by the comment that follows and add the else case
; to support representation as assembler strings.
(define (defstringaux charlist)
  (if (null? charlist)
      (seq)
      (if #t ; (member (car charlist) '(#\" #\( #\) #\cr #\newline #\space))
	  (seq (data byte (char->integer (car charlist)))
	       (defstringaux (cdr charlist))))))
	  
	  

; Defines a label in the string segment after which the data for the
; string is placed and returns the label.
(define (defstring str)
  (let ((l (gen-label)))
    (addcode! 'stringseg
	      (seq l
		   (defstringaux (string->list str))))
    l))

	    

; ---------------------------------------------------------------------------
; Console I/O
; ---------------------------------------------------------------------------

(define (PUTC . rest)
  (let ((r (if (null? rest) #f (car rest))))
    (seq (if r (inst 'mov al r) (seq))
	 (call 'putc))))

(define (wait-for-char)
  (pushing (list 'aX)
	   (call 'cgetc)))

; Attempt to unify all the _Pr??_ routines from as86.m4

(define (pr rest)
  (apply seq (map pr-one rest)))

; The register saves should be verified, there might be too many or not enough.
(define (pr-one x)
  (cond ((string? x)
	 (pushing (list 'ax)
		  (mov (r?? si)	(defstring 
				  (string-append x (integer->char 0))))
		  (call 'puts)))
	((char? x)
	 (pushing (list 'ax)
		  (PUTC (char->integer x))))
	((list? x)
	 (let ((type (car x))
	       (args (cdr x)))
	   (case type
	     ((asciiz stringz) (pushing (list 'ax)
					(mov (r?? si) (car args))
					(call 'puts)))
	     ((addr address) (pushing (list 'ax es)
				      (pushm (car args) (cadr args))
				      (popm es (r?? di))
				      (call 'putaddr)))
	     ((byte) (pushing (list 'ax)
			      (mov al (car args))
			      (call 'puthexbyte)))
	     ((word) (pushing (list 'ax)
			      (mov ax (car args))
			      (call 'puthexword)))
	     ((dword long) (pushing (list 'ad)
				    (mov eax (car args))
				    (call 'puthexdword)))
	     ((zone bzone) (pushing (list 'ax)
				    (ppmov es (car args))
				    (mov (r?? di) (cadr args))
				    (mov (r?? cx) (caddr args))
				    (call 'putzone)))
	     ((wzone) (pushing (list 'ax)
			       (ppmov es (car args))
			       (mov (r?? di) (cadr args))
			       (mov (r?? cx) (caddr args))
			       (call 'putwzone)))
	     ((lzone) (pushing (list 'ad)
			       (ppmov es (car args))
			       (mov (r?? di) (cadr args))
			       (mov (r?? cx) (caddr args))
			       (call 'putlzone)))
	     ((regs) (call 'dumpregs))
	     ((eregs) (if p-mode 
			  (call 'dumperegs)
			  (call 'rm_dumperegs))))))))

; ---------------------------------------------------------------------------
; Debugging
; ---------------------------------------------------------------------------
				    
; .........................
; Heavy debugging
; .........................

; Identifying position by value of gs in dumps.
(define (STEP x)
  (if debug2 
      (ppmov gs x)
      (seq)))


(define (STOP seg vidmembase x y char attr)
  (if debug2
      (seq (if (equal? seg ds)
	       (seq)
	       (inst 'seg seg))
	   (mov (op-size word (index vidmembase (+ (* 2 x) (* 2 80 y))))
		(+ char (* 256 attr))))))

(define (STOP2 seg vidmembase x y char attr char2 attr2)
  (if debug2
      (seq (if (equal? seg ds)
	       (seq)
	       (inst 'seg seg))
	   (mov (op-size word (index vidmembase (+ (* 2 x) (* 2 80 y))))
		(+ char (* 256 (+ attr (* 256 (+ char2 (* 256 attr2))))))))))

(define debug-message '())
(define debug-x 0)
(define debug-y 0)
(define debug-color #x09)
(define debug-seg fs)
(define debug-beg #xB8000)

(define (DMSG mes x y color seg beg)
  (set! debug-message (string->list mes))
  (set! debug-x x)
  (set! debug-y y)
  (set! debug-color color)
  (set! debug-seg seg)
  (set! debug-beg beg))

(define (letter1)
  (if (null? debug-message)
      (error "No more letters to display.")
      (let ((code (STOP debug-seg debug-beg debug-x debug-y 
			(char->integer (car debug-message)) debug-color)))
	(set! debug-message (cdr debug-message))
	(set! debug-x (+ 1 debug-x))
	(if (>= debug-x 80) (begin (set! debug-x (- debug-x 80))
				   (set! debug-y (+ debug-y 1)))))))

(define (letter2)
  (if (or (null? debug-message) (null? (cdr debug-message)))
      (error "No more letters to display.")
      (let ((code (STOP2 debug-seg debug-beg debug-x debug-y 
			 (char->integer (car debug-message)) debug-color
			 (char->integer (cadr debug-message)) debug-color)))
	(set! debug-message (cddr debug-message))
	(set! debug-x (+ 2 debug-x))
	(if (>= debug-x 80) (begin (set! debug-x (- debug-x 80))
				   (set! debug-y (+ debug-y 1)))))))
      
(define (letter n)
  (if (= n 0)
      (seq)
      (if (>= n 2) 
	  (seq (letter2) (letter (- n 2)))
	  (seq (letter1) (letter (- n 1))))))

(define (DBG msg)
  (set! debug-message (append (string->list msg) debug-message))
  (letter (string-length msg)))

(define (SST . rest)
  (if debug2 (wait-for-char) (seq)))
		      
				    
; ---------------------------------------------------------------------------
; Support for install-time options dichotomy
; ---------------------------------------------------------------------------

(define OPTION_SIG "%%InStTiMeOpTiOn%%")

(define (IT_OPT str part1 . rest)
  (let ((part2 (if (null? rest) #f (car rest)))
	(opt1 (gen-label))
	(opt2 (gen-label))
	(end (gen-label)))
    (seq (inst 'push (op-size dword (codeaddr opt1)))
	 (inst 'ret)
	 (data long (codeaddr opt1))
	 (data long (codeaddr opt2))
	 (defstringaux (string->list 
			(string-append OPTION_SIG 
				       str 
				       (char->string (integer->char 0)))))
	 opt1
	 (if mode32 (seq) (inst 'add sp 2))
	 part1
	 (if part2 (inst 'jmp end) (seq))
	 opt2
	 (if mode32 (seq) (inst 'add sp 2))
	 part2
	 end)))


; ---------------------------------------------------------------------------
; Library
; ---------------------------------------------------------------------------

; A function that retreives the required function from the library.
; The returned program used "mode" to know how to return and args
; as actual parameters for the formal parameters that appear in the
; library entry.
(define (lib-fetch name mode . args)
  (let ((entry (assoc name lib-alist)))
    (if entry
	(apply ((cdr entry) mode) args)
	(error "Can't find this entry in library :" name))))

(define lib-alist (list))

(define (lib-add! name fun)
  (set! lib-alist (cons (cons name fun) lib-alist)))

; A return instruction that depends on the "calling" mode.
(define (ret mode)
  (case mode
    ((inline) (seq))
    ((far) (inst 'retf))
    ((near) (inst 'ret))
    (else (error "Invalid mode: " mode))))



; .........................
; detecting CPU type
; .........................

(lib-add! 'testCPU
	  ; Procedure initially taken from Robert L. Hummel's book
	  ; on 80x86 programming, then enhanced with the help of Cedric Ware.
	  ;
	  ; AH
	  ; 01	8088 or 8086
	  ; 02	80286
	  ; 03	80386
	  ; 04	486
	  ; 05	Pentium
	  ; 06	P6
	  ;
	  ; AL
	  ; 00	No FPU
	  ; 01	8087
	  ; 02	80287
	  ; 03	80387
	  ; 04	486DX or 487SX
	  ; 05	Pentium
	  ; 06	P6
	  ;
	  ; Note: 0400 <=> 486SX w/o FPU.
	  ;
	  ; destroys all general purpose registers, including FP registers
	  (lambda (mode)
	    (lambda ()
	      (seq 
	       (inst 'mov dx #x0100) ; default answer
	       (inst 'mov ax -1)   ; non zero
	       (inst 'push ax)   ; was [NDP_STATUS],-1 ;
	       (inst 'mov bp sp)   ; save SP
	       ; on 8088/8086, bits 12 to 15 of pushed Flags value are set,
	       ; that's why we'll try to clear them.
	       (inst 'pushf)
	       (inst 'pop ax)
	       (inst 'and ah #x0f)   ;(clear most significant bits)
	       (inst 'push ax)
	       (inst 'popf)
	       (inst 'pushf)
	       (inst 'pop ax)
	       (inst 'and ah #xf0)   ;(get them back)
	       (inst 'cmp ah #xf0)
	       (inst 'jz (label 'cpuid_2))   ;if all set, it's a 8086/8088
	       ; on the 80286, on the opposite, they are cleared
	       (inst 'inc dh)
	       (inst 'pushf)
	       (inst 'pop ax)
	       (inst 'or ah #xf0)   ;(set most significant bits)
	       (inst 'push ax)
	       (inst 'popf)
	       (inst 'psuhf)
	       (inst 'pop ax)
	       (inst 'and ah #xf0)   ;(get them back)
	       (inst 'jz (label 'cpuid_2))   ;if cleared, it's a 80286
	       ; On the 80486, the EFLAGS registers has got an AC alignment bit
	       ; From late 486s on, the EFLAGS register has an ID bit 
	       ; (authorize CPUID)
	       (inst 'inc dh)
	       (inst 'cli)
	       (inst 'and sp -4) ; -4 for (not 3) I hope that's right (P.P.)
	                         ; align sp
	       (inst 'pushfd)
	       (inst 'pop eax)
	       (inst 'mov ebx eax)   ; save eflags
	       (inst 'xor eax (bitwise-or FLAG_AC FLAG_ID))
	       (inst 'push eax)
	       (inst 'popfd)
	       (inst 'pushfd)
	       (inst 'pop eax)
	       (inst 'push ebx)
	       (inst 'popfd)   ; restore AC
	       (inst 'mov sp bp)   ; restore sp
	       (inst 'xor eax ebx)   ; look if AC or ID was modified
	       (inst 'sti)
	       (inst 'jz (label 'cpuid_2))   ; no: it's a 386
	       (inst 'inc dh)
	       (inst 'test eax FLAG_ID)   ; ID modified ?
	       ;(!) I'm not sure this works like this:
	       ;perhaps instead I should test FLAG_ID...
	       (inst 'jz (label 'cpuid_2))   ; no, it's an early 486
	       ; Now, we can use the CPUID instruction to determine the
	       ; CPU type
	       (clr eax)   ; sequence smaller than mov eax,1 !
	       (inst 'inc eax)
	       (data byte #x0F)   ; These 2 lines were 1 before (P.P.)
	       (data byte #xA2)   ; I hope it still works. CPUID
	       (inst 'and ah 7)   ; CPU type in eax bits [11:8]
	       (inst 'mov dh ah)   ; stored in dh
	       ; Now, test for coprocessor:
	       (label 'cpuid_2)
	       (inst 'fninit)   ;reset status word
	       (inst 'gnstsw (index bp))
	       (inst 'cmp (op-size byte (index bp)) 0)   ;if non zero, no FPU
	       (inst 'jnz (label 'cpuid_3))
	       (inst 'fnstcw (index bp))   ;try to write control word
	       (inst 'and (op-size word (index bp)) #x103f)
	       (inst 'cmp (op-size word (index bp)) #x003f)
	       (inst 'jnz (label 'cpuid_3))
	       ; The only case when the FPU doesn't answer is a 80386+80287
	       (inst 'mov dl dh)   ;initialize FPU type
	       (inst 'cmp dh 3)   ;no 386 => ok
	       (inst 'jnz (label 'cpuid_3))
	       ; 387/287: the initialized NDP must manage infinity,
	       ; but they're the same to the 287	FLD1
	       (inst 'fldz)
	       (inst 'fdiv (reg 'st) (reg 'st))
	       (inst 'fld (reg 'st))
	       (inst 'fchs)
	       (inst 'fcompp)
	       (inst 'fstsw (index bp))
	       (inst 'mov ax (index bp))
	       (inst 'sahf)
	       (inst 'jnz (label 'cpuid_3))   ;different => 387
	       (inst 'dec dl)
	       (label 'cpuid_3)
	       (inst 'mov ax dx)
	       (ret mode)))))



; ---------------------------------------------------------------------------
; Memory management in protected mode
; ---------------------------------------------------------------------------


; see file "mem386.doc" for dox...

; .........................
; A20 line control
; .........................

; Constants to access line A20
(define KB_command_port #x64)  ;IOaddr to write to
(define KB_status_port  #x64)  ;IOaddr to read from
(define KB_data_port    #x60)  ;IOaddr to read from
(define IB_free         #x02)  ;status bitmask: is kbd controller ready ?
(define KB_command_A20  #xD1)  ;command bitmask to access A20 line.
(define set_A20_ON      #xDF)
(define set_A20_OFF     #xDD)

; .........................
; Macros to control line A20
; .........................

(lib-add! 'kb_ready
	  ; Test if kbd port is available (needed before to send it commands)
	  ; - {$1} must be a label where to jump if line not available
	  ; - {$2} must be the number of tries before to jump to {$1}
	  ; ** destroys AX,CX
	  (lambda (mode)
	    (lambda (fail-label . rest)
	      (let ((tries (if (null? rest) 100 (car rest)))
		    (retry-label (gen-label)))
		(seq (if (equal? tries (r?? cx))
			 (seq)
			 (inst 'mov  (r?? cx) tries))
		     retry-label
		     (inst 'in al KB_status_port)
		     (inst 'test al IB_free)
		     (inst 'loopne retry-label)
		     (inst 'jcxz fail-label)
		     (ret mode))))))


(lib-add! 'set_a20
	  ; (de)activivates line A20,
	  ; - {$1} is "ON" or "OFF",
          ;    or absent (then ah contains A20_ON or A20_OFF)
	  ; - (!) keyboard must be ready (use previous macro for that)
	  ; ** destroys AX
	  (lambda (mode)
	    (lambda rest
	      (let ((state (if (null? rest) #f (car rest))))
		(seq (if state
			 (inst 'mov ah (if (equal? state "ON")
					   set_A20_ON
					   set_A20_OFF))
			 (seq))
		     (inst 'mov al KB_command_A20)
		     (inst 'out KB_command_port al)
		     ; Linus does call KB_Ready again !
		     (inst 'mov al ah)
		     (inst 'out KB_command_port al)
		     ; Linus does call KB_Ready again !
		     (ret mode))))))
			       
(lib-add! 'test_a20
	  ; * tests if line A20 is activated, by seeing whether addresses are
	  ;  wrapped over 1MB.
	  ; * sets ax iff the line is activated
	  (lambda (mode)
	    (lambda ()
	      (let ((exit-label (gen-label)))
		(seq (pushm ds es 'fd)
		     (inst 'cli) ; disable interrupts
		     (clr ax)
		     (inst 'mov es ax) ; es=$0000
		     (inst 'dec ax)
		     (inst 'mov ds ax) ; ds=$FFFF
		     (inst 'mov ax (index -2)) ; verify [$10FFEE]=[$0FFEE]
		     (inst 'seg es)
		     (inst 'cmp ax (index -18))
		     (inst 'stc)
		     (inst 'jnz exit-label)
		     (inst 'inc (index -2)) ; verify [$10FFEE]=[$0FFEE]
		                            ; after modification
		     (inst 'seg es)
		     (inst 'cmp ax (index -18))
		     (inst 'mov (index -2) ax)
		     (inst 'mov ax -1)
		     (inst 'jz exit-label)
		     (inst 'inc ax)
		     exit-label
		     (popm ds es 'fd) ; enable interrupts again if needed
		     (ret mode))))))


; .........................
; NMI enabling/disabling
; .........................

(lib-add! 'disable_NMI
	  (lambda (mode)
	    (lambda ()
	      (seq (inst 'mov al #x80)
		   (inst 'out #x70 al)
		   (inst 'in al #x71)
		   (ret mode)))))

(lib-add! 'enable_NMI
	  (lambda (mode)
	    (lambda ()
	      (seq (inst 'mov al #x00)
		   (inst 'out #x70 al)
		   (inst 'in al #x71)
		   (ret mode)))))

; "...to accomplish a warm boot equivalent to Ctrl-Alt-Del, store 1234h
; in 0040h:0072h and jump to FFFFh:0000h.    For a cold boot equivalent
; to a reset, store 0000h at 0040h:0072h before jumping."

(lib-add! 'reset_computer
	  (lambda (mode)
	    (lambda ()
	      (seq (inst 'mov al #xF0)
		   (inst 'out #x64 al)))))

(define (defdescriptor p1 p2 p3 p4)
  ;  base,limit,_P_DPL_DT_Type,_G_D_0_AVL_0000
  (seq (data word (modulo p2 (<< 1 16)))
       (data word (modulo p1 (<< 1 16)))
       (data byte (modulo (>> p1 16) (<< 1 8)))
       (data byte p3)
       (data byte (+ p4 (modulo (>> p2 16) (<< 1 4))))
       (data byte (>> p1 24))))

(define (defgate p1 p2 p3 p4)
  ;  sel,ofs,000compteur,_P_DPL_DT_Type
  (seq (data word (modulo p2 (<< 1 16)))
       (data word p1)
       (data byte p3)
       (data byte p4)
       (data word (>> p2 16))))

(lib-add! 'set-descriptor 
	  ;	DESCRIPT
	  ; initialize a descriptor with
	  ; - {$1} being the base address of the descriptor to initialize
	  ; - ebx is base address of the segment
	  ; - ecx is the limit field in byte or page according to granularity
	  ; - al has got fields: P,DPL,DT,Type
	  ; - ah has got fields: G,D,AVL
	  ; * the routine saves all registers
	  (lambda (mode)
	    (lambda (r)
	      (seq (pushing (list ebx ecx)
			    (inst 'mov (index r) cx)
			    (inst 'mov (index r 2) bx)
			    (inst 'shr ecx 16)
			    (inst 'shr ebx 16)
			    (inst 'or cl ah)
			    (inst 'mov ch bh)
			    (inst 'mov bh al)
			    (inst 'mov (index r 6) cx)
			    (isnt 'mov (index r 4) bx))
		   (ret mode)))))
(lib-add! 'get-descriptor
	  ;	DESCRIPT
	  ; This macro does the converse of preceding one:
	  ; - {$1} contains base address of the descriptor
	  ; * returns in ebx the base address of the segment
	  ; * returns in ecx the limit field
	  ; * returns in al fields: P,DPL,DT,Type
	  ; * returns in ah fields: G,D,AVL
	  (lambda (mode)
	    (lambda (r)
	      (seq (inst 'mov cx (index r 6))
		   (inst 'mov bx (index r 4))
		   (inst 'mov al bh)
		   (inst 'mov bh ch)
		   (inst 'mov ah cl)
		   (inst 'and ah #xF0)
		   (inst 'and cx #x0F)
		   (inst 'shl ebx 16)
		   (inst 'shl ecx 16)
		   (inst 'mov bx (index r 2))
		   (inst 'mov cx (index r))
		   (ret mode)))))


; ---------------------------------------------------------------------------
; Constants and structures
; ---------------------------------------------------------------------------

; .........................
; Segment selectors
; .........................

(define SEL_MASK	#xFFF8)	; offset to descriptor entry in table
(define SEL_TI		#x0004)	; bit cleared:GDT set:LDT
(define SEL_RPL		#x0003)	; priviledge of segment making request


; .........................
; Segment descriptors
; .........................

(define DT_G		#x80) ; Granularity of limit 0yte 1=page
(define DT_D		#x40) ; Default size using CS/SS 0=16bit 1=32bit
(define DT_AVL		#x10) ; AVaiLable to OS (unused by CPU)
(define DT_P		#x80) ; segment Present bit
(define DT_PL0		#x00) ; descriptor PriviLedge
(define DT_PL1		#x20)
(define DT_PL2		#x40)
(define DT_PL3		#x60)
(define DT_DT		#x10) ; Descriptor Type, 1: segment, 0: special
; if DT_Segment (bit DT) is set:
(define DT_A		#x1) ; Accessed
(define DT_Data		#x0)
(define DT_Writable	#x2) ; Writable
(define DT_E		#x4) ; Extensible downward
(define DT_Code		#x8)
(define DT_R		#x2) ; Readable
(define DT_C		#x4) ; Conforming
; if DT_Segment -bit DT) is cleared:
(define DT_TSS16_A	#x1) ; available TSS 16
(define DT_LDT		#x2) ; LDT
(define DT_TSS16_B	#x3) ; busy TSS 16
(define DT_CallGate16	#x4)
(define DT_TaskGate	#x5)
(define DT_IntGate16	#x6)
(define DT_TrapGate16	#x7)
(define DT_TSS32_A	#x9) ; available TSS 32
(define DT_TSS32_B	#xB) ; busy TSS 32
(define DT_CallGate32	#xC)
(define DT_IntGate32	#xE)
(define DT_TrapGate32	#xF)
; a Trap gate requires CPL<=IOPL to be called; IF is unchanged
; an Interrupt gate clears IF; both Trap and Interrupt gates clear NT
; the counter is the number of stack arguments of a call gate.

; .........................
; Page table entries
; .........................

(define PG_P		#x01) ; Present
(define PG_RW		#x02) ; Read/Write
(define PG_US		#x04) ; User (PL3) access possible
(define PG_PWT		#x08) ; Page Write Through (486+)
(define PG_PCD		#x10) ; Page Cache Disable (486+)
(define PG_A		#x20) ; Accessed (486+)
(define PG_D		#x40) ; Dirty (undefined for page directory entries)
; bits 7,8 reserved
; bits 9-11 available to OS (typically, page locking, copy-on-write, etc)


; .........................
; Structures
; .........................

(define std-struc
  (seq
   (struc 'xmsmp ; XMS Memory move
	  ; Struc XMS_Move_Params   ;; parameters for XMS func 0Bh
	  (elem 'lengthtomove long)
	  (elem 'sourcehandle word)
	  (elem 'sourceoffset long)
	  (elem 'desthandle word)
	  (elem 'destoffset long))
   (apply struc
	  (cons 'pushad 
		(list-elems long 'edi 'esi 'ebp 'esp 'ebx 'edx 'ecx 'eax)))
   (apply struc 
	  (cons 'pusha
		(list-elems word 'di 'si 'bp 'sp 'bx 'dx 'cx 'ax)))
   (apply struc 
	  (cons 'v86int 
		(list-elems long 'edi 'esi 'ebp 'aesp 'ebx 'edx 'ecx 'eax
			         'eip 'cs 'eflags 'esp 'ss 'es 'ds 'fs 'gs)))
   (apply struc 
	  (cons 'intstack
		(list-elems long 'eip 'cs 'eflags 'esp 'ss)))
   (apply struc 
	  (cons 'tss16
		(list-elems word 'backlinkselector 
			    'sp_cpl0 'ss_cpl0 'sp_cpl1 'ss_cpl1 'sp_cpl2 
			    'ss_cpl2 'ip 'flags 'ax 'cx 'dx 'bx 'sp 'bp 
			    'si 'di 'es 'cs 'ss 'ds 'ldt)))
   (apply struc 
	  (cons 'tss32
		(append
		 (list-elems word 'backlinkselector 
			     'esp_cpl0 'ss_cpl0 'esp_cpl1 'ss_cpl1 'esp_cpl2 
			     'ss_cpl2 'cr3 'eip 'eflags 'eax 'ecx 'edx 'ebx
			     'esp 'ebp 'esi 'edi 'es 'cs 'ss 'ds 'fs 'gs 'ldt)
		 (list (elem 'TrapBit word)
		       (elem 'IO_Map_Base word)))))))

; TSS32_CR3 is reserved by Intel but everyone I know seem to use it as
; a place to store the CR3 page directory register for the given task.






; ===========================================================================
; Translation of "i386.m4"
; ===========================================================================
; This is (was) the m4 include file to expand generic programs
; into as86 sources for the i386 subproject.
; See ~moose/info/i386impl.info for more info about this.
; ===========================================================================

; ---------------------------------------------------------------------------
; Structures
; ---------------------------------------------------------------------------

(define api-struc
  (seq (struc 'as                          ; Signature_Struc
	      (elem 'magic_number long)    ; magic number from magic file
	      (elem 'object_name byte 20)
	      (elem 'version byte 8))
       ; API_Header_Struc
       ; size is 32. offset is 32 from beginning of API file
       ; (32 first bytes are signature and version)
       (struc 'ah
	      (elem 'totallength long)	;total length of the system file
	      (elem 'entrypoint long)	;address of object to return
	      (elem 'offset long)	;offset where to load it
	      (elem 'headersize long)	;size of header fields in file
	      (elem 'expandedsize long) ;size when expanded to memory
	      (elem 'pad long)		;fill in
	      (elem 'checksum long)	;checksum
	      (elem 'longchecksum long)) ;long checksum
       ; AerialThread,  These threads are lighter than light: Aerial !
       (struc 'at
	      (elem 'save_ebx long)	;initial Top of Stack
	      (elem 'save_esi long)	;Data Stack Pointer
	      (elem 'save_edi long)	;Return Stack Pointer
	      (elem 'save_ebp long)	;Those two are reserved for the
	      (elem 'save_esp long)	;threading mechanism.
	      (elem 'save_resp long)	;ESP save when going ret-threaded
	      (elem 'state long)	;flags used for thread sync'
	      (elem 'max_esi long)	;bottom of data stack
	      (elem 'max_edi long)	;bottom of return stack
	      (elem 'linkback long))	;Link Back to thread object
       ))

; ---------------------------------------------------------------------------
; Some values
; ---------------------------------------------------------------------------

(define BYTEBITS 8)    	        ;bits in a byte (smallest addressable value)
(define CHARABITS 0)   		;bits of char in address
(define WORDABITS 1)   		;bits of word in address
(define CELLABITS 2)   		;bits of cell in address
(define PAGEABITS 12)  		;bits of page in address

(define CELL CELLSIZE)	        ;a cell is 4 bytes long
(define MAX_INT (- (<< 1 31) 1));MAX integer
(define MIN_INT (- (<< 1 31)))	;MIN integer


; ******** There's more to translate from that file ***********

(define (Loopno0 . rest)
  (let ((beg (gen-label)))
    (seq beg
	 (apply seq rest)
	 (inst 'loop beg))))

(define (Loop . rest)
  (let ((beg (gen-label))
	(end (gen-label)))
    (seq (inst 'jecxz end)
	 beg
	 (apply seq rest)
	 (inst 'loop beg)
	 end)))

; ===========================================================================
; Translation of "biosapi.S"
; ===========================================================================
; the original TUNES module to run on raw ISA PCs
; see $TUNES/doc/info/BIOS-API.nfo
; ===========================================================================


; ---------------------------------------------------------------------------
; Version identification
; ---------------------------------------------------------------------------

(define API_MAGIC MAGIC_PC_API)
(define API_NAME "OSLm i386 BIOS API")

; ---------------------------------------------------------------------------
; Constants
; ---------------------------------------------------------------------------

(define APISEG 	0x0000)		; Where the API is originally loaded
(define APIOFS 	PAGESIZE)	; The corresponding offset in PM.
(define IRQ_INT_BASE 	0x90)		; IRQ's are mapped at INT 90h
(define IDT_SIZE 	0xA0*8)		; Is that enough ?
(define GDT_SIZE 	0x1000)		; Is that enough ?
(define COMMAND_LINE 	0x800)		; Where to put the command line
(define V86_SERVER_INT 0xFD)		; V86 server interrupt
(define PM_STACK_SIZE 	0x4000)		; 4096 entries

; ---------------------------------------------------------------------------
; Some options
; ---------------------------------------------------------------------------

define({__IO_Delay__},
	{locals({D1},{D2},{jmp D1}__nl__{D1:	jmp D2}__nl__{D2:})})dnl


; ---------------------------------------------------------------------------
; Standard PM Selectors
; ---------------------------------------------------------------------------

;Name			sel.	base lim.	description
; These are valid even when paging is disabled
; thus, linear memory must be identity-mapped to physical memory,
; or at least the PMPART of the API

(define NULLSEG		#x0000)	; Invalid null segment
(define TEMPSEG		#x0008)	; Temporary
(define ALLMEMCS	#x0010)	; Whole memory (code), both
(define ALLMEMDS	#x0018)	; Whole memory (data), both
(define RMPARTCS	#x0020)	; RM Part of the API (code), noPG/IdPG
(define RMPARTDS	#x0028)	; RM Part of the API (data), noPG/IdPG
(define PMPARTCS	#x0030)	; PM Part of the API (code), noPG/IdPG
(define PMPARTDS	#x0038)	; PM Part of the API (data), noPG/IdPG

(define KL0CS		#x0040)	; Kernel PL0 Code Segment, both w/ trick
(define KL0DS		#x0048)	; Kernel PL0 Data Segment, both w/ trick
(define KL1CS		#x0051)	; Kernel PL1 Code Segment, PG
(define KL1DS		#x0059)	; Kernel PL1 Data Segment, PG
(define KL2CS		#x0062)	; Kernel PL2 Code Segment, PG
(define KL2DS		#x006A)	; Kernel PL2 Data Segment, PG
(define KL3CS		#x0073)	; Kernel PL3 Code Segment, PG
(define KL3DS		#x007B)	; Kernel PL3 Data Segment, PG

(define KL0CG		#x0080)	; Kernel PL0 Call Gate
(define KL1CG		#x0089)	; Kernel PL1 Call Gate
(define KL2CG		#x0092)	; Kernel PL2 Call Gate
(define KL3CG		#x009B)	; Kernel PL3 Call Gate

(define KTSS		#x00A0)	; Kernel TSS, PG
(define DummyTSS	#x00A8)	; dummy for non-saving "switches"
(define V86TSS		#x00B0)	; V86 TSS, PG
(define UserTSS		#x00B8)	; User TSS, PG

; What about adding more tasks, or task gates ?

(define Orig_GDT_SIZE (* 8 #x00C0))

(define SysCallArgC	8)	; Number of arguments to a "system call"


; ---------------------------------------------------------------------------
; Segmenting
; ---------------------------------------------------------------------------

; .text is rmpart
; .data is pmpart
; .bss is pmpart udata.   Could anyone tell me what "bss" stands for ??????

; after all, everything is .text, so as86 is not confused



; List of segments. Each segment is a pair. The car is the name (a symbol).
; The cdr is an alist between numbers and lists of code fragments.
; The numbers define the order in which the fragments will be outputed.
; By default, add-code! adds to the end of part number 0 but can also
; add at the end of any other existing or new numbered part.
(define segment-list '())

(add-new-segment! 'definitions)
(add-code! '(definitions -1) (text ".org APIOFS"))

(add-new-segment! 'rminitcode)
(add-code! '(rminitcode -1) (seq (comment "Segment: rminitcode")
				 (text ".align 16")
				 (label 'rminitcode_beg)))
(add-code! '(rminitcode 1) (seq (text ".align 16")
				(label 'rminitcode_end)))

(add-new-segment! 'rminitdata)
(add-code! '(rminitdata -1) (seq (comment "Segment: rminitdata")
				 (text ".align 16")
				 (text ".text")
				 (label 'rminitdata_beg)))
(add-code! '(rminitdata 1) (seq (text ".align 16")
				(label 'rminitdata_end)))

(add-new-segment! 'rmcode)
(add-code! '(rmcode -1) (seq (comment "Segment: rmcode")
			     (text ".text")
			     (text ".align 16")
			     (label 'rmcode_beg)))
(add-code! '(rmcode 1) (seq (text ".align 16")
			    (label 'rmcode_end)))

(add-new-segment! 'rmstringseg)
(add-code! '(rmstringseg -1) (seq (comment "Segment: rmstringseg")
				  (text ".align 16")
				  (label 'rmstringseg_beg)))
(add-code! '(rmstringseg 1) (seq (text ".align 16")
				 (label 'rmstringseg_end)))

(add-new-segment! 'rmdata)
(add-code! '(rmdata -1) (seq (comment "Segment: rmdata")
			     (text ".align 16")
			     (label 'rmdata_beg)))
(add-code! '(rmdata 1) (seq (text ".align 16")
			    (label 'rmdata_end)))

(add-new-segment! 'rmudata)
(add-code! '(rmudata -1) (seq (comment "Segment: rmudata")
			      (text ".text")
			      (text ".align 16")
			      (label 'rmudata_beg)))
(add-code! '(rmudata 1) (seq (text ".align 16")
			     (label 'rmudata_end)))

(add-new-segment! 'pminitcode)
(add-code! '(pminitcode -1) (seq (comment "Segment: pminitcode")
				 (text ".align 16")
				 (label 'pminitcode_beg)))
(add-code! '(pminitcode 1) (seq (text ".align 16")
				(label 'pminitcode_end)))

(add-new-segment! 'pminitdata)
(add-code! '(pminitdata -1) (seq (comment "Segment: pminitdata")
				 (text ".align 16")
				 (label 'pminitdata_beg)))
(add-code! '(pminitdata 1) (seq (text ".align 16")
				(label 'pminitdata_end)))

(add-new-segment! 'pmcode)
(add-code! '(pmcode -1) (seq (comment "Segment: PM code")
			     (text ".align 16")
			     (label 'pmcode_beg)))
(add-code! '(pmcode 1) (seq (text ".align 16")
			    (label 'pmcode_end)))

(add-new-segment! 'pmstringseg)
(add-code! '(pmstringseg -1) (seq (comment "Segment: pmstringseg")
				  (text ".text")
				  (text ".align 16")
				  (label 'pmstringseg_beg)))
(add-code! '(pmstringseg 1) (seq (text ".align 16")
				 (label 'pmstringseg_end)))

(add-new-segment! 'symbolseg)
(add-code! '(symbolseg -1) (seq (comment "Segment: symbolseg")
				(text ".align 16")
				(label 'symbolseg_beg)))
(add-code! '(symbolseg 1) (seq (text ".align 16")
			       (label 'symbolseg_end)))

(add-new-segment! 'pmdata)
(add-code! '(pmdata -1) (seq (comment "Segment: pmdata")
			     (text ".align 16")
			     (label 'pmdata_beg)
			     (text ".text")))
(add-code! '(pmdata 1) (seq (text ".align 16")))
(add-code! '(pmdata 2) (seq (text ".align 16")
			    (label 'pmdata_end)))

(add-new-segment! 'pmudata)
(add-code! '(pmudata -1) (seq (comment "Segment: pmudata")
			      (text ".align 16")
			      (label 'pmudata_beg)))
(add-code! '(pmudata 1) (seq (text ".align 16")))
(add-code! '(pmudata 2) (seq ((label 'pmudata_end))))


; Create a new segment. The order of creation is the order of output.
(define (add-new-segment! segment-name)
  (if (assoc segment-name segment-list)
      (error "There already is such a segment :" segment-name)
      (set! segment-list (append segment-list (cons segment-name (list))))))

; Adds codes to a segment in the segment list
; By default, add-code! adds to the end of part number 0 but can also
; add at the end of any other existing or new numbered part.
(define (add-code! segment-name-or-list code)
  (let ((segment (assoc segment-name segment-list))
	(segment-name (if (list? segment-name-or-list)
			  (car segment-name-or-list)
			  segment-name-or-list))
	(segment-part-number (if (list? segment-name-or-list)
				 (cadr segment-name-or-list)
				 0)))
    (if segment
	(letrec ((alist (cdr segment))
		 (addpart (lambda (alist num code)
			    (if (null? alist)
				(list (cons num code))
				(if (> (caar alist) num)
				    (cons (cons num (list code)) alist)
				    (if (= (caar alist) num)
					(cons (cons (caar alist)
						    (append (cdar alist) code))
					      (cdr alist))
					(cons (car alist)
					      (addpart 
					       (cdr alist) num code))))))))
	  (set-cdr! segment (addpart alist segment-part-number code)))
	(if (member segment-name '(codeseg dataseg udataseg stringseg))
	    (add-code! (list (auto-segment segment-name) segment-part-number)
		       code)
	    (error "There is no such segment :" segment-name)))))

; Chooses the segment from the modes and the supplied segment category
(define (auto-segment sym)
  (if p-mode
      (if init-mode
	  (case sym
	    ((codeseg) 'pminitcode)
	    ((dataseg udataseg stringseg) 'pminitdata))
	  (case sym
	    ((codeseg) 'pmcode)
	    ((dataseg) 'pmdata)
	    ((udataseg) 'pmudata)
	    ((stringseg) 'pmstringseg)))
      (if init-mode
	  (case sym
	    ((codeseg) 'rminitcode)
	    ((dataseg udataseg stringseg) 'rminitdata))
	  (case sym
	    ((codeseg) 'rmcode)
	    ((dataseg) 'rmdata)
	    ((udataseg) 'rmudata)
	    ((stringseg) 'rmstringseg)))))

; Returns the program consisting of every part of every segment
; as a tree of sequences
(define (all-segments)
  (apply seq (map (lambda (seg)
		    (apply seq (map (lambda (num-list-pair)
				      (apply seq (cdr num-list-pair)))
				    (cdr seg))))
		  segment-list)))
    

(define (pmaddr x)
  (if (number? x)
      (+ pmofs x)
      (binop "+" pmofs x)))
(define (rmaddr x)
  (if (number? x)
      (+ rmofs x)
      (binop "+" rmofs x)))
(define (initaddr x) 	x)

(define (codeaddr x) (if p-mode (pmaddr x) (rmaddr x)))
(define (stringaddr x) (if p-mode (pmaddr x) (rmaddr x)))




; ---------------- Some constants -----------------

(add-code! 'definitions
  (seq
   (def 'rminitcode_len 
	(binop "-" (name 'rminitcode_end) (name 'rminitcode_beg)))
   (def 'rminitdata_len  
	(binop "-" (name 'rminitdata_end) (name 'rminitdata_beg)))
   (def 'rmcode_len       
	(binop "-" (name 'rmcode_end) (name 'rmcode_beg)))
   (def 'rmdata_len       
	(binop "-" (name 'rmdata_end) (name 'rmdata_beg)))
   (def 'rmudata_len      
	(binop "-" (name 'rmudata_end) (name 'rmudata_beg)))
   (def 'rmstringseg_len  
	(binop "-" (name 'rmstringseg_end) (name 'rmstringseg_beg)))

   (def 'pminitcode_len   
	(binop "-" (name 'pminitcode_end) (name 'pminitcode_beg)))
   (def 'pminitdata_len   
	(binop "-" (name 'pminitdata_end) (name 'pminitdata_beg)))
   (def 'pmcode_len       
	(binop "-" (name 'pmcode_end) (name 'pmcode_beg)))
   (def 'pmdata_len       
	(binop "-" (name 'pmdata_end) (name 'pmdata_beg)))
   (def 'pmudata_len      
	(binop "-" (name 'pmudata_end) (name 'pmudata_beg)))
   (def 'pmstringseg_len  
	(binop "-" (name 'pmstringseg_end) (name 'pmstringseg_beg)))
   (def 'symbolseg_len    
	(binop "-" (name 'symbolseg_end) (name 'symbolseg_beg)))

   (def 'pmpart_beg       (name 'pmcode_beg))
   (def 'pmpart_end       (name 'pmudata_end))
   (def 'pmpart_len       
	(binop "-" (name 'pmpart_end) (name 'pmpart_beg)))
   (def 'pmpart_actbeg    (name 'pminitcode_beg))
   (def 'pmpart_actend    (name 'pmdata_end))
   (def 'pmpart_actlen    
	(binop "-" (name 'pmpart_actend) (name 'pmpart_actbeg)))
   (def 'pmpart_bsslen    (name 'pmudata_len))

   (def 'rmpart_beg       (name 'rmcode_beg))
   (def 'rmpart_end       (name 'rmudata_end))
   (def 'rmpart_len       
	(binop "-" (name 'rmpart_end) (name 'rmpart_beg)))
   (def 'rmpart_actbeg    (name 'rminitcode_beg))
   (def 'rmpart_actend    (name 'rmdata_end))
   (def 'rmpart_actlen    
	(binop "-" (name 'rmpart_actend) (name 'rmpart_actbeg)))
   (def 'rmpart_bsslen    (name 'rmudata_len))

   (def 'total_length     
	(binop "+" (name 'rmpart_actlen) (name 'pmpart_actlen)))

   (def 'pmofs    (binop "-" 0 (name 'pmpart_end)))
   (def 'rmofs    (binop "-" #x10000 (name 'rmpart_end)))

   (def 'APISIZE (name 'total_length))))

(add-code! 'definitions
	   (seq std-struc api-struc))


; ---------------------------------------------------------------------------
; Code generation
; ---------------------------------------------------------------------------

(if debug1
    (begin (load "debug.scm")
	   (load "rmdebug.scm")))

(load "header.scm")
(load "head.scm")
(load "rmpart.scm")
(load "pmpart.scm")
(load "xrv86.scm")

(load "video.scm")
;;(load "vp.scm")

(load "paging.scm")	; (!) must be last in list

(compile (all-segments) 'as86 "lll.s")
































; ===========================================================================
;What I'm not sure if or how to translate and other things to do:

;define({setf},{ifelse(isreg($1),1,{or $1,$1},{test $1,#-1})})
;Isn't it mov instead of test ?

;define({__unseg__},         in as86.m4
;what does that do? it was used in the push/pop stuff and I neglected it.

;movm used __mov__ which did a lot of not so obviously efficient optimization
;I used the instruction mov instead of doing __mov__.

;define({SegOfsToLinear},
;what does that do?

;newheap & allot have not been translated. Can't we use the struc instead ?

;dnl ---------------- Exportation de donnees -----------------------
;what does that do?

;function puts depends on lsa which is in exportation de donnees

; check testcpu for occurences of P.P. and verify that the modifications
; are valid

;dnl -------------- Support for install-time options dichotomy ----------------
;what does that do?

;To do, fix the immediate or not mode (# or not) for numbers in the
; translation from the internal representation to as86
; (difficult because of rol eax,16 and mov eax,#0)

; ===========================================================================

; In the following, aren't the use16 ad use32 useless ?
; And I still have to deal with these heaps.
;__pmcode__({
;;;; SEGMENT: PM CODE
;.align 16
;use32
;pmcode_beg:
;})
;__pmudata__({
;;;; SEGMENT: PM UDATA
;pmudata_beg:
;newheap({u},pmudata_beg,0)
;})
;__pminitcode__({
;;;; SEGMENT: PM INIT CODE
;.align 16
;use32
;pminitcode_beg:
;})
;__rmcode__({
;;;; SEGMENT: RM CODE
;dnl .text
;use16
;.align 16
;rmcode_beg:
;})
;__rmudata__({
;;;; SEGMENT: RM UDATA
;dnl .text
;.align 16
;rmudata_beg:
;newheap({ru},rmudata_beg,0)
;})
;__rminitcode__({
;;;; SEGMENT: RM INIT CODE
;.align 16
;use16
;rminitcode_beg:
;})

; And what about these : (the separate "end" diversions and this heap stuff)
;__pmdata__(
;{
;.align 16
;})
;__pmdataend__(
;{
;.align 16
;pmdata_end:
;})
;__pmudata__(
;{
;alignheap(u,16)
;})
;__pmudataend__(
;{
;pmudata_end = alallot(u,16,0)
;})
;__rmudata__(
;{
;rmudata_end = alallot(ru,16,0)
;})


; ===========================================================================
