; Generic object manipulation facilities
; ===========================================================================

; Creates an object constructor from a type.
; An object is just a pair where the car is the type 
; and the cdr is a list of the object members.
; If a second argument is passed, it must be a list
; with one element for each member. That element
; is irrelevent if there is no default value and it is
; the default value if there is one. Supply the
; list of defaults fixes the number of members.
; By default there are any number of members.
(define (constructor t . rest)
  (if (null? rest)
      (lambda args 
	(cons t args))
      (lambda args
	(cons t (append args (list-tail (car rest) (length args)))))))

; From a type, creates a fonction that determines if an
; object is of that type.
(define (detector t)
  (lambda (object) 
    (and (pair? object) 
	 (eq? (car object) t))))

; General constructor.
; (define (make t . args) ((constructor t) args))
; Or more efficiently, without the test for the presence of the type
(define make list)

; General detector.
(define (is-a? t object) ((detector t) object))

; Extracts the type.
(define (type object) 
  (if (pair? object)
      (car object)
      (if (number? object)
	  'number
	  (error "Tried to get type of unsupported object :" object))))

; Takes n and makes a getter which retreives the n-th element.
; 0 is the first object member, not the type.
(define (getter n)
  (lambda (object)
    (list-ref (cdr object) n)))

; Returns the object members.
(define object-members cdr)

; Returns the object size.
(define (object-size object)
  (length (object-members object)))

