; A set of misc. tools
; ===========================================================================

; Makes a relative path global
(define (global x) (string-append path x))

; A relative path version of load
(define (include x) (load (global x)))

; Reload everything
(define (l) (load (global "lll.scm")))

; Shortcut for MIT-Scheme
(define (r) (restart 1))

; Something to make loose ends obvious
(define (******)
  (error "Execution has reached the unimplemented expression ******"))

(define (compile prog target file)
  (call-with-output-file (global file)
    (lambda (port)
      (display-asm prog target port))))

; ---------------------------------------------------------------------------
; A few high-order function
; ---------------------------------------------------------------------------

; Fold right. Ex: (foldr 0 + '(3 2 6)) = (+ 3 (+ 2 (+ 6 0)))
(define (foldr zero fun l)
  (if (null? l)
      zero
      (fun (car l) (foldr zero fun (cdr l)))))

; Fold right wihtout zero. Ex: (foldr1 + '(3 2 6)) = (+ 3 (+ 2 6))
(define (foldr1 fun l)
  (if (null? (cdr l))
      (car l)
      (fun (car l) (foldr1 fun (cdr l)))))

; Fold left.
(define (foldl zero fun l)
  (foldr zero (lambda (x y) (fun y x)) (reverse l)))

; Fold left without zero.
(define (foldl1 fun l)
  (foldr (lambda (x y) (fun y x)) (reverse l)))

; Curry 
; Passes only the first argument and returns a lambda that awaits the others.
(define (curry f x)
  (lambda (rest)
    (apply f (cons x rest))))

; ---------------------------------------------------------------------------
; A few mathematical extentions
; ---------------------------------------------------------------------------

; returns (quotient n 2^i)  (right-shift)
(define (>> n i)
  (if (= i 0) n (>> (quotient n 2) (- i 1))))

; returns (* n 2^i)  (left-shift)
(define (<< n i)
  (if (= i 0) n (<< (* n 2) (- i 1))))

(define (bitwise-or2 a b)
  (if (= a 0)
      b
      (+ (max (modulo a 2) (modulo b 2))
	 (* 2 (bitwise-or2 (quotient a 2) (quotient b 2))))))

(define (bitwise-or args)
  (foldr1 bitwise-or2 args))

(define (bitwise-and2 a b)
  (if (= a 0)
      0
      (+ (min (modulo a 2) (modulo b 2))
	 (* 2 (bitwise-and2 (quotient a 2) (quotient b 2))))))

(define (bitwise-and args)
  (foldr1 bitwise-and2 args))

(define (not a number-of-bits)
  (if (= number-of-bits 0)
      0
      (+ (- 1 (modulo a 2))
	 (* 2 (not (quotient a 2) (- number-of-bits 1))))))

(define not32 (curry not 32))
(define not16 (curry not 16))
(define not8 (curry not 8))

; ---------------------------------------------------------------------------
; Error messages and error handling
; ---------------------------------------------------------------------------

(define (assert predicate object)
  (if (predicate object)
      object
      (Error "Object failed assertion :" object)))

(define unsupported-target
  "This construct doesn't support the specified target language.")

